/* FILE: mlog.c                                 (D. Tottingham  03/30/90)

This is a collection of C functions that manage the log file for xdetect.
All functions have been written and compiled medium model.  The following
functions are included:

o_check_pathname ()             check log pathname for validity
o_initialize_params ()          initialize log pathname
o_open_logfile ()               open a log file for output
o_set_pathname ()               set log pathname
o_write_logfile ()              write string to log file stream

EXTERNAL FUNCTIONS CALLED:

er_abort ()                     display an error message then quit
u_build_date_fn ()              build a date filename: pathname\YRMNDY
u_build_timeline ()             convert abs. time to an ascii string
u_timestamp ()                  get timestamp

HISTORY:
   none

*/


/*************************************************************************
                            INCLUDE FILES

*************************************************************************/
#include <stdio.h>
#include <string.h>
#include <sys\types.h>
#include <sys\timeb.h>
#include <time.h>

#include "mconst.h"
#include "merror.h"
#include "mlog.h"
#include "mutils.h"


/*************************************************************************
                                GLOBALS

*************************************************************************/
PRIVATE char log_filename[MAX_FILENAME_LENGTH * 2];
PRIVATE char log_pathname [MAX_FILENAME_LENGTH];


/*=======================================================================*
 *                              build_log_fn                             *
 *=======================================================================*/
/* Build a log filename as such: YRMNDY.LOG.                             */

PRIVATE
void build_log_fn ()
{
   struct timeb time;
   double abs_time;
   int j;

   ftime (&time);
   abs_time = u_convert_time (time);
   j = u_build_date_fn ( abs_time, log_pathname, log_filename);

   j += sprintf (log_filename+j, ".LOG");
}

/*=======================================================================*
 *                          o_initialize_params                          *
 *=======================================================================*/
/* Initialize log pathname.                                              */

PUBLIC
void o_initialize_params ()
{
   strncpy (log_pathname, LOG_PATHNAME, MAX_FILENAME_LENGTH);
}

/*=======================================================================*
 *                             o_check_pathname                          *
 *=======================================================================*/
/* Check pathname for validity.                                          */

PUBLIC
void o_check_pathname ()
{
   long ds;
   int drive;

   /* Do we have any diskspace? */
   if ((ds = u_get_diskspace (log_pathname, &drive)) == 0)
      er_abort (O_DISK_FULL);
   else if (ds == INVALID_DRIVE)
      er_abort (O_INVALID_DRIVE);
}

/*=======================================================================*
 *                              o_open_logfile                           *
 *=======================================================================*/
/* Open a log file for output.                                           */

PUBLIC
FILE * o_open_logfile ()
{
   FILE * stream;
   char * timeline;

   if (LOG_FILE) {
      build_log_fn ();
      if ((stream = fopen (log_filename, "a")) == NULL)
         er_abort (O_INVALID_LOGFILE);
      timeline = u_build_timeline (u_timestamp(), 0);
      fprintf (stream, "\n*** %14.12s ***\n", &timeline[11]);
      free (timeline);
      return (stream);
   } else return (NULL);
}

/*=======================================================================*
 *                              o_set_pathname                           *
 *=======================================================================*/
/* Set log pathname.                                                     */

PUBLIC
void o_set_pathname (pathname)
char pathname[];
{
   unsigned int length;

   strncpy (log_pathname, pathname, (MAX_FILENAME_LENGTH - 1));
   length = strlen(log_pathname);
   if (log_pathname[length-1] != ':' && log_pathname[length-1] != '\\')
      strcat (log_pathname, "\\");
}

/*=======================================================================*
 *                              o_write_logfile                          *
 *=======================================================================*/
/* Write string to log file stream.                                      */

PUBLIC
void o_write_logfile (str_ptr)
char * str_ptr;
{
   FILE * log_stream;

   if (LOG_FILE) {
      log_stream = o_open_logfile();
      fprintf (log_stream, "%s", str_ptr);
      fclose (log_stream);
   }
}
